"use strict";
//
// Copyright 2021 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProtocolAddress = exports.Pni = exports.Aci = exports.ServiceId = void 0;
const Native = require("../Native");
const uuid = require("uuid");
var ServiceIdKind;
(function (ServiceIdKind) {
    ServiceIdKind[ServiceIdKind["Aci"] = 0] = "Aci";
    ServiceIdKind[ServiceIdKind["Pni"] = 1] = "Pni";
})(ServiceIdKind || (ServiceIdKind = {}));
const SERVICE_ID_FIXED_WIDTH_BINARY_LEN = 17;
class ServiceId extends Object {
    // This has to be public for `InstanceType<T>`, which we use below.
    constructor(serviceIdFixedWidthBinary) {
        super();
        if (serviceIdFixedWidthBinary.length != SERVICE_ID_FIXED_WIDTH_BINARY_LEN) {
            throw new TypeError('invalid Service-Id-FixedWidthBinary');
        }
        this.serviceIdFixedWidthBinary = serviceIdFixedWidthBinary;
    }
    static fromUuidBytesAndKind(uuidBytes, kind) {
        const buffer = Buffer.alloc(SERVICE_ID_FIXED_WIDTH_BINARY_LEN);
        buffer[0] = kind;
        buffer.set(uuidBytes, 1);
        return new this(buffer);
    }
    getServiceIdBinary() {
        return Native.ServiceId_ServiceIdBinary(this.serviceIdFixedWidthBinary);
    }
    getServiceIdFixedWidthBinary() {
        return Buffer.from(this.serviceIdFixedWidthBinary);
    }
    getServiceIdString() {
        return Native.ServiceId_ServiceIdString(this.serviceIdFixedWidthBinary);
    }
    toString() {
        return Native.ServiceId_ServiceIdLog(this.serviceIdFixedWidthBinary);
    }
    downcastTo(subclass) {
        // Omitting `as object` results in TypeScript mistakenly assuming the branch is always taken.
        if (this instanceof subclass) {
            return this;
        }
        throw new TypeError(`expected ${subclass.name}, got ${this.constructor.name}`);
    }
    static parseFromServiceIdFixedWidthBinary(serviceIdFixedWidthBinary) {
        let result;
        switch (serviceIdFixedWidthBinary[0]) {
            case ServiceIdKind.Aci:
                result = new Aci(serviceIdFixedWidthBinary);
                break;
            case ServiceIdKind.Pni:
                result = new Pni(serviceIdFixedWidthBinary);
                break;
            default:
                throw new TypeError('unknown type in Service-Id-FixedWidthBinary');
        }
        return result.downcastTo(this);
    }
    static parseFromServiceIdBinary(serviceIdBinary) {
        const result = ServiceId.parseFromServiceIdFixedWidthBinary(Native.ServiceId_ParseFromServiceIdBinary(serviceIdBinary));
        return result.downcastTo(this);
    }
    static parseFromServiceIdString(serviceIdString) {
        const result = this.parseFromServiceIdFixedWidthBinary(Native.ServiceId_ParseFromServiceIdString(serviceIdString));
        return result.downcastTo(this);
    }
    getRawUuid() {
        return uuid.stringify(this.serviceIdFixedWidthBinary, 1);
    }
    getRawUuidBytes() {
        return Buffer.from(this.serviceIdFixedWidthBinary.buffer, 1);
    }
    isEqual(other) {
        return this.serviceIdFixedWidthBinary.equals(other.serviceIdFixedWidthBinary);
    }
}
exports.ServiceId = ServiceId;
class Aci extends ServiceId {
    static fromUuid(uuidString) {
        return this.fromUuidBytes(uuid.parse(uuidString));
    }
    static fromUuidBytes(uuidBytes) {
        return this.fromUuidBytesAndKind(uuidBytes, ServiceIdKind.Aci);
    }
}
exports.Aci = Aci;
class Pni extends ServiceId {
    static fromUuid(uuidString) {
        return this.fromUuidBytes(uuid.parse(uuidString));
    }
    static fromUuidBytes(uuidBytes) {
        return this.fromUuidBytesAndKind(uuidBytes, ServiceIdKind.Pni);
    }
}
exports.Pni = Pni;
class ProtocolAddress {
    constructor(handle) {
        this._nativeHandle = handle;
    }
    static _fromNativeHandle(handle) {
        return new ProtocolAddress(handle);
    }
    static new(name, deviceId) {
        if (typeof name !== 'string') {
            name = name.getServiceIdString();
        }
        return new ProtocolAddress(Native.ProtocolAddress_New(name, deviceId));
    }
    name() {
        return Native.ProtocolAddress_Name(this);
    }
    /**
     * Returns a ServiceId if this address contains a valid ServiceId, `null` otherwise.
     *
     * In a future release ProtocolAddresses will *only* support ServiceIds.
     */
    serviceId() {
        try {
            return ServiceId.parseFromServiceIdString(this.name());
        }
        catch (_a) {
            return null;
        }
    }
    deviceId() {
        return Native.ProtocolAddress_DeviceId(this);
    }
    toString() {
        return `${this.name()}.${this.deviceId()}`;
    }
}
exports.ProtocolAddress = ProtocolAddress;
//# sourceMappingURL=Address.js.map